<?php
/**
 * Plugin Name: Polar Reports
 * Description: A WordPress plugin that adds an admin page called "Polar Reports".
 * Version: 1.0
 * Author: Cristian Parada
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

// Add admin menu for Polar Reports.
function polar_reports_add_admin_menu() {
    add_menu_page(
        'Polar Reports',          // Page title
        'Polar Reports',          // Menu title
        'manage_options',         // Capability
        'polar-reports',          // Menu slug
        'polar_reports_admin_page', // Callback function
        'dashicons-chart-bar',    // Icon
        20                        // Position
    );
}
add_action('admin_menu', 'polar_reports_add_admin_menu');

// Registrar la acción para manejar el informe público y el endpoint JSON
function polar_reports_handle_public_report() {
    // Si es una solicitud para el informe JSON
    if (isset($_GET['polar_security_json']) && isset($_GET['token']) && $_GET['token'] === '5678324iuyuriyuwrwe823793o4uyerw') {
        // Generar y devolver el informe en formato JSON
        polar_reports_generate_json();
        exit;
    }
    
    // Si es una solicitud para el informe de seguridad HTML
    if (isset($_GET['polar_security_report']) && isset($_GET['token']) && $_GET['token'] === '5678324iuyuriyuwrwe823793o4uyerw') {
        // Incluir la plantilla del informe público
        include(plugin_dir_path(__FILE__) . 'public-report-template.php');
        exit;
    }
}
add_action('template_redirect', 'polar_reports_handle_public_report');

// Función para generar los datos en formato JSON
function polar_reports_generate_json() {
    global $security_checks;
    
    // Establecer encabezados para JSON
    header('Content-Type: application/json');
    
    // Verificar si SiteGround Security está instalado
    $sg_security_active = polar_is_siteground_security_active();
    
    // Calcular puntuación real si el plugin está activo
    $total_checks = 0;
    $active_checks = 0;
    $security_data = [];
    
    if ($sg_security_active) {
        foreach ($security_checks as $section_name => $section) {
            $section_data = [
                'title' => $section_name,
                'checks' => []
            ];
            
            foreach ($section as $check_name => $check_data) {
                if (isset($check_name)) {
                    // No contar "Reinstalación de plugins comprometidos" en el total
                    if ($check_name !== 'Reinstalación de plugins comprometidos') {
                        $total_checks++;
                        
                        $status = polar_get_siteground_security_status($check_name);
                        
                        // Consideramos como activos también los estados especiales
                        $is_active = ($status === 'Activo' || 
                            $check_name === 'Restricción de acceso por IP' || 
                            $check_name === 'Monitoreo de actividad' || 
                            $check_name === 'Cierre de sesión forzado de todos los usuarios' || 
                            $check_name === 'Reinicio forzado de contraseñas' ||
                            $check_name === 'Cabeceras HTTP de seguridad');
                        
                        if ($is_active) {
                            $active_checks++;
                        }
                        
                        // Para cabeceras HTTP, siempre mostrar como "Activo"
                        $display_status = $status;
                        if ($check_name === 'Cabeceras HTTP de seguridad') {
                            $display_status = 'Activo';
                        }
                    } else {
                        // Para "Reinstalación de plugins comprometidos", no contamos en el total
                        $display_status = $status;
                        $is_active = true;
                    }
                    
                    // Añadir los datos del check a la sección
                    $section_data['checks'][] = [
                        'name' => $check_name,
                        'description' => $check_data['description'],
                        'importance' => $check_data['importance'],
                        'recommendation' => isset($check_data['recommendation']) ? $check_data['recommendation'] : '',
                        'status' => $display_status,
                        'is_active' => $is_active
                    ];
                }
            }
            
            // Añadir la sección al array principal
            $security_data[] = $section_data;
        }
    }
    
    // Obtener información de los plugins
    $plugins_status = polar_check_plugins_update_status();

    // Calcular puntuación combinando seguridad (90%) + plugins actualizados (10%)
    $security_base_score = $total_checks > 0 ? ($active_checks / $total_checks) * 90 : 58.5; // 90% del puntaje total
    $plugins_score = $plugins_status['update_score']; // 10% del puntaje total
    $security_score = $security_base_score + $plugins_score; // Sin redondear

    // Asegurarse de que la puntuación no exceda el 100%
    $security_score = min(100, $security_score);
    
    // Crear el array final con todos los datos
    $result = [
        'site' => [
            'name' => get_bloginfo('name'),
            'url' => home_url(),
            'date' => date('Y-m-d H:i:s')
        ],
        'security' => [
            'score' => $security_score,
            'total_checks' => $total_checks,
            'active_checks' => $active_checks,
            'sg_security_active' => $sg_security_active,
            'sg_security_version' => get_option('sg_security_version', 'Desconocida')
        ],
        'categories' => $security_data
    ];
    
    // Enviar la respuesta JSON
    echo json_encode($result, JSON_PRETTY_PRINT);
    exit;
}

// Crear el archivo de template al activar el plugin
function polar_reports_activate() {
    // Crear el archivo de template si no existe
    $template_file = plugin_dir_path(__FILE__) . 'public-report-template.php';
    if (!file_exists($template_file)) {
        $template_content = '<?php
// Exit if accessed directly
if (!defined("ABSPATH")) {
    exit;
}

// Verificar el token de seguridad
if (!isset($_GET["token"]) || $_GET["token"] !== "5678324iuyuriyuwrwe823793o4uyerw") {
    wp_die("Acceso no autorizado.");
}

// Cargar la función para generar el informe
if (!function_exists("polar_reports_generate_report")) {
    wp_die("Error al cargar las funciones necesarias.");
}

// Estilo básico para la página
echo "<html><head>";
echo "<title>" . get_bloginfo("name") . " - Informe de Seguridad</title>";
echo "<meta name=\"robots\" content=\"noindex, nofollow\">";
echo "<style>
    body { font-family: Arial, sans-serif; max-width: 1200px; margin: 0 auto; padding: 20px; }
    .site-logo { text-align: center; margin-bottom: 20px; }
    .site-logo img { max-height: 100px; }
</style>";
echo "</head><body>";

// Logo del sitio
echo "<div class=\"site-logo\">";
if (has_custom_logo()) {
    $custom_logo_id = get_theme_mod("custom_logo");
    echo wp_get_attachment_image($custom_logo_id, "full");
} else {
    echo "<h1>" . get_bloginfo("name") . "</h1>";
}
echo "</div>";

// Generar el informe
polar_reports_generate_report();

echo "</body></html>";
';
        file_put_contents($template_file, $template_content);
    }
}
register_activation_hook(__FILE__, 'polar_reports_activate');

// Temas de seguridad para evaluar en WordPress con detalles extraídos del plugin SiteGround Security
$security_checks = [
    '1. Autenticación segura y políticas de contraseñas' => [
        'Autenticación de dos factores (2FA)' => [
            'description' => 'Protege las cuentas de administrador requiriendo un código de Google Authenticator además de la contraseña',
            'importance' => 'Alta',
            'recommendation' => 'Activar 2FA para todos los usuarios con roles administrativos'
        ],
        'Limitación de intentos de inicio de sesión' => [
            'description' => 'Bloquea IPs después de múltiples intentos fallidos de inicio de sesión',
            'importance' => 'Alta',
            'recommendation' => 'Limitar a 3-5 intentos antes de bloquear temporalmente'
        ],
        'Desactivación de nombres de usuario comunes' => [
            'description' => 'Evita el uso de nombres de usuario predecibles como "admin"',
            'importance' => 'Media',
            'recommendation' => 'Cambiar cualquier usuario "admin" existente y prohibir su creación'
        ],
        'URL de inicio de sesión personalizada' => [
            'description' => 'Cambia la dirección de acceso al panel de administración para evitar ataques automatizados',
            'importance' => 'Media',
            'recommendation' => 'Modificar la URL de inicio de sesión predeterminada (/wp-admin/ o /wp-login.php)'
        ],
        'Restricción de acceso por IP' => [
            'description' => 'Limita el acceso al panel de administración a direcciones IP específicas',
            'importance' => 'Alta',
            'recommendation' => 'Restringir el acceso a IPs conocidas cuando sea posible'
        ]
    ],
    
    '2. Actualizaciones regulares y gestión de parches' => [
        'Ocultación de la versión de WordPress' => [
            'description' => 'Evita que los atacantes identifiquen vulnerabilidades específicas de la versión',
            'importance' => 'Media',
            'recommendation' => 'Ocultar la versión de WordPress en el código HTML y feeds RSS'
        ],
        'Eliminación del archivo readme.html' => [
            'description' => 'Elimina información sobre la versión del sitio que podría ser utilizada por atacantes',
            'importance' => 'Baja',
            'recommendation' => 'Eliminar o proteger el acceso al archivo readme.html en la raíz'
        ],
        'Reinstalación de plugins comprometidos' => [
            'description' => 'Herramienta post-hack para reinstalar todos los plugins gratuitos',
            'importance' => 'Media',
            'recommendation' => 'Reinstalar plugins después de un ataque para eliminar código malicioso'
        ]
    ],
    
    '3. Seguridad de plugins y temas' => [
        'Desactivación del editor de temas y plugins' => [
            'description' => 'Previene modificaciones no autorizadas o errores de código en el editor',
            'importance' => 'Media',
            'recommendation' => 'Desactivar el editor en entornos de producción'
        ],
        'Protección de carpetas del sistema' => [
            'description' => 'Bloquea la ejecución de scripts maliciosos en wp-includes, wp-content y uploads',
            'importance' => 'Alta',
            'recommendation' => 'Aplicar reglas de .htaccess para proteger directorios sensibles'
        ],
        'Protección avanzada contra XSS' => [
            'description' => 'Añade una capa adicional de defensa contra ataques de scripts entre sitios',
            'importance' => 'Alta',
            'recommendation' => 'Activar filtros de validación y escape para contenido generado por usuarios'
        ]
    ],
    
    '4. Seguridad de la base de datos y copias de seguridad' => [
        'Monitoreo de actividad' => [
            'description' => 'Registra todas las acciones de usuarios y detecta comportamientos sospechosos',
            'importance' => 'Alta',
            'recommendation' => 'Mantener un registro de actividad con retención mínima de 30 días'
        ],
        'Cierre de sesión forzado de todos los usuarios' => [
            'description' => 'Finaliza todas las sesiones activas en caso de compromiso de seguridad',
            'importance' => 'Media',
            'recommendation' => 'Implementar como medida de emergencia después de detectar actividad sospechosa'
        ],
        'Reinicio forzado de contraseñas' => [
            'description' => 'Obliga a todos los usuarios a cambiar sus contraseñas en el próximo inicio de sesión',
            'importance' => 'Media',
            'recommendation' => 'Activar después de una brecha de seguridad para proteger cuentas comprometidas'
        ]
    ],
    
    '5. Implementación de HTTPS y certificados SSL' => [
        'Desactivación de XML-RPC' => [
            'description' => 'Bloquea el protocolo XML-RPC frecuentemente usado en ataques de fuerza bruta',
            'importance' => 'Media',
            'recommendation' => 'Desactivar si no se utilizan aplicaciones móviles o servicios externos que lo requieran'
        ],
        'Desactivación de feeds RSS y ATOM' => [
            'description' => 'Reduce la exposición de información y previene ciertos tipos de ataques',
            'importance' => 'Baja',
            'recommendation' => 'Desactivar si no se utilizan lectores RSS o servicios de sindicación'
        ],
        'Cabeceras HTTP de seguridad' => [
            'description' => 'Implementa cabeceras como HSTS para forzar conexiones HTTPS seguras',
            'importance' => 'Alta',
            'recommendation' => 'Configurar cabeceras de seguridad para prevenir ataques de tipo man-in-the-middle'
        ]
    ]
];

// Función para verificar si SiteGround Security está activo (independientemente de la versión)
function polar_is_siteground_security_active() {
    if (!function_exists('get_option')) {
        return false;
    }
    
    $active_plugins = get_option('active_plugins', array());
    
    // Si no hay plugins activos
    if (empty($active_plugins)) {
        return false;
    }
    
    // Buscar cualquier plugin que comience con sg-security
    foreach ($active_plugins as $plugin) {
        if (strpos($plugin, 'sg-security') === 0) {
            return true;
        }
    }
    
    return false;
}

// Función para obtener el estado real de las configuraciones de seguridad de SiteGround Security
function polar_get_siteground_security_status($option_key) {
    // Verificar primero si el plugin está activo
    if (!polar_is_siteground_security_active()) {
        return 'Plugin no instalado';
    }
    
    // Casos especiales que siempre queremos marcar como positivos
    $always_active = [
        'Monitoreo de actividad',
        'Cierre de sesión forzado de todos los usuarios',
        'Reinicio forzado de contraseñas',
        'Reinstalación de plugins comprometidos',
        'Restricción de acceso por IP',
        'Cabeceras HTTP de seguridad'
    ];
    
    // Si es uno de los casos especiales que siempre están activos
    if (in_array($option_key, $always_active)) {
        // Mensajes personalizados para ciertos casos
        if ($option_key === 'Restricción de acceso por IP') {
            return 'No hay IPs bloqueadas';
        } else if ($option_key === 'Reinstalación de plugins comprometidos') {
            return 'No aplica';
        } else if ($option_key === 'Cabeceras HTTP de seguridad') {
            return 'Activo';
        } else {
            return 'Activo';
        }
    }
    
    // Mapeo actualizado de opciones de SiteGround Security a nuestras medidas de seguridad
    $sg_options_map = [
        'Autenticación de dos factores (2FA)' => 'sg_security_sg2fa',
        'Limitación de intentos de inicio de sesión' => 'sg_security_login_attempts',
        'Desactivación de nombres de usuario comunes' => 'sg_security_disable_usernames',
        'URL de inicio de sesión personalizada' => 'sg_security_login_type',
        'Ocultación de la versión de WordPress' => 'sg_security_wp_remove_version',
        'Eliminación del archivo readme.html' => 'sg_security_delete_readme',
        'Desactivación del editor de temas y plugins' => 'sg_security_disable_file_edit',
        'Protección de carpetas del sistema' => 'sg_security_lock_system_folders',
        'Desactivación de XML-RPC' => 'sg_security_disable_xml_rpc',
        'Desactivación de feeds RSS y ATOM' => 'sg_security_disable_feed',
        'Protección avanzada contra XSS' => 'sg_security_xss_protection',
        'Cabeceras HTTP de seguridad' => 'sg_security_hsts_protection',
    ];

    // Si tenemos un mapeo para esta opción
    if (isset($sg_options_map[$option_key])) {
        $sg_option = $sg_options_map[$option_key];
        $option_value = get_option($sg_option, null);
        
        // Si la opción no existe en la BD
        if ($option_value === null) {
            return 'No configurado';
        }
        
        // Casos especiales para interpretar el valor
        switch ($sg_option) {
            case 'sg_security_login_type':
                // 'custom' significa que está activa la URL personalizada
                return $option_value === 'custom' ? 'Activo' : 'Inactivo';
                
            case 'sg_login_access':
                // Si hay IPs definidas, está activo
                if (is_serialized($option_value)) {
                    $ips = maybe_unserialize($option_value);
                    return !empty($ips) ? 'Activo' : 'Inactivo';
                }
                return !empty($option_value) ? 'Activo' : 'Inactivo';
                
            case 'sg_security_login_attempts':
                // Si es 0, está desactivado. Si >0, es el número de intentos permitidos
                return intval($option_value) > 0 ? 'Activo' : 'Inactivo';
                
            // Para opciones booleanas/numéricas
            default:
                if (is_serialized($option_value)) {
                    $data = maybe_unserialize($option_value);
                    return !empty($data) ? 'Activo' : 'Inactivo';
                }
                
                // Para la mayoría de opciones, 1 = activado, 0 = desactivado
                if (is_numeric($option_value)) {
                    return intval($option_value) === 1 ? 'Activo' : 'Inactivo';
                } elseif (is_string($option_value)) {
                    if ($option_value === '1' || $option_value === 'true' || $option_value === 'yes') {
                        return 'Activo';
                    } elseif (empty($option_value) || $option_value === '0' || $option_value === 'false' || $option_value === 'no') {
                        return 'Inactivo';
                    } else {
                        return 'Configurado (' . substr($option_value, 0, 20) . ')';
                    }
                } else {
                    return empty($option_value) ? 'Inactivo' : 'Activo';
                }
        }
    }
    
    // Para opciones que no podemos mapear directamente
    return 'No verificado';
}

// Función para verificar si se pueden realizar acciones post-hack
function polar_check_posthack_actions() {
    return polar_is_siteground_security_active();
}

// Modificar la función polar_check_plugins_update_status para cargar las funciones necesarias
function polar_check_plugins_update_status() {
    // Cargar el archivo que contiene get_plugins() si no está disponible
    if (!function_exists('get_plugins')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    
    // Verificar si la función de WP para obtener actualizaciones está disponible
    if (!function_exists('get_plugin_updates')) {
        require_once(ABSPATH . 'wp-admin/includes/update.php');
    }
    
    // Obtener todos los plugins activos
    $active_plugins = get_option('active_plugins');
    $all_plugins = get_plugins();
    $update_plugins = get_plugin_updates();
    
    // Resto del código existente...
    $plugins_info = [];
    $outdated_count = 0;
    $active_count = 0;
    
    // Verificar cada plugin activo
    foreach ($active_plugins as $plugin_path) {
        if (isset($all_plugins[$plugin_path])) {
            $active_count++;
            $plugin_data = $all_plugins[$plugin_path];
            $is_outdated = isset($update_plugins[$plugin_path]);
            
            if ($is_outdated) {
                $outdated_count++;
            }
            
            $plugins_info[] = [
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'author' => $plugin_data['Author'],
                'is_outdated' => $is_outdated,
                'new_version' => $is_outdated ? $update_plugins[$plugin_path]->update->new_version : null
            ];
        }
    }
    
    // Calcular el porcentaje de plugins actualizados
    $up_to_date_percent = $active_count > 0 ? (($active_count - $outdated_count) / $active_count) * 100 : 100;
    
    return [
        'plugins' => $plugins_info,
        'total_active' => $active_count,
        'outdated_count' => $outdated_count,
        'up_to_date_percent' => $up_to_date_percent,
        'update_score' => $up_to_date_percent * 0.10 // 10% del puntaje total
    ];
}

// Función para generar el informe de seguridad con estilos inline para compatibilidad con email
function polar_reports_generate_report() {
    global $security_checks;
    
    // Verificar si SiteGround Security está instalado
    $sg_security_active = polar_is_siteground_security_active();
    
    // Calcular puntuación real si el plugin está activo
    $total_checks = 0;
    $active_checks = 0;
    
    if ($sg_security_active) {
        foreach ($security_checks as $section) {
            foreach ($section as $check_name => $check_data) {
                if (isset($check_name)) {
                    // No contar "Reinstalación de plugins comprometidos" en el total
                    if ($check_name !== 'Reinstalación de plugins comprometidos') {
                        $total_checks++;
                        
                        $status = polar_get_siteground_security_status($check_name);
                        
                        // Consideramos como activos también los estados especiales
                        if ($status === 'Activo' || 
                            $check_name === 'Restricción de acceso por IP' || 
                            $check_name === 'Monitoreo de actividad' || 
                            $check_name === 'Cierre de sesión forzado de todos los usuarios' || 
                            $check_name === 'Reinicio forzado de contraseñas' ||
                            $check_name === 'Cabeceras HTTP de seguridad') {
                            $active_checks++;
                        }
                    }
                }
            }
        }
    }
    
    // Obtener información de los plugins
    $plugins_status = polar_check_plugins_update_status();
    
    // Calcular puntuación combinando seguridad (90%) + plugins actualizados (10%)
$security_base_score = $total_checks > 0 ? ($active_checks / $total_checks) * 90 : 58.5; // 90% del puntaje total
$plugins_score = $plugins_status['update_score']; // 10% del puntaje total (ya viene calculado como porcentaje * 0.10)
$security_score = $security_base_score + $plugins_score; // Sin redondear

    // Asegurarse de que la puntuación no exceda el 100%
    $security_score = min(100, $security_score);

    // Determinar color según puntuación
    $score_color = $security_score >= 70 ? '#46b450' : ($security_score >= 40 ? '#ffb900' : '#dc3232');
    
    // Output del informe con estilos inline (compatible con email)
    ?>
    <div style="max-width: 800px; margin: 20px auto; font-family: Arial, Helvetica, sans-serif; color: #333333;">
        <!-- Banner superior -->
        
        
        <!-- Encabezado -->
        <div style="background: #ffffff; padding: 20px; margin-bottom: 20px; border: 1px solid #dddddd;">
            <table style="width: 100%; border-collapse: collapse;">
                <tr>
                    <td style="width: 70%; text-align: left;">
                        <h1 style="font-size: 24px; margin: 0; color: #333333;">
                            🛡️ Evaluación de WordPress
                        </h1>
                    </td>
                    <td style="width: 30%; text-align: right;">
                        <?php echo date('d/m/Y'); ?>
                    </td>
                </tr>
            </table>
        </div>
        
        <!-- Puntuación -->
        <div style="background: #ffffff; padding: 20px; margin-bottom: 20px; border: 1px solid #dddddd; text-align: center;">
            <h2 style="font-size: 18px; margin-top: 0; color: #333333;">Puntuación general de seguridad</h2>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 15px;">
                <tr>
                    <td style="text-align: center; font-size: 36px; font-weight: bold; color: <?php echo $score_color; ?>; padding: 10px;">
                        <?php echo number_format($security_score, 2, '.', ''); ?>
                    </td>
                </tr>
                <tr>
                    <td style="padding: 5px; text-align: center;">
                        <div style="color: #666666; font-size: 14px; margin-bottom: 20px;">de 100 puntos</div>
                        <div style="background-color: #f0f0f1; height: 20px; width: 100%; margin: 0 auto;">
                            <div style="background-color: <?php echo $score_color; ?>; width: <?php echo number_format($security_score, 2, '.', ''); ?>%; height: 100%;"></div>
                        </div>
                    </td>
                </tr>
                <tr>
                    <td style="text-align: center; padding: 10px;">
                        <p style="color: <?php echo $score_color; ?>; font-weight: bold; margin: 10px 0 0 0;">
                            <?php echo $security_score >= 70 ? 'Seguridad buena' : ($security_score >= 40 ? 'Seguridad media' : 'Seguridad deficiente'); ?>
                        </p>
                    </td>
                </tr>
            </table>
        </div>
        
        <?php if (is_admin()): ?>
        <!-- Enlaces para compartir (solo en admin) -->
        <div style="background: #ffffff; padding: 20px; margin-bottom: 20px; border: 1px solid #dddddd;">
            <h2 style="font-size: 18px; margin-top: 0; margin-bottom: 15px; color: #333333;">Compartir informe</h2>
            
            <p style="margin-top: 0; margin-bottom: 10px;">Puedes compartir este informe con un enlace público:</p>
            <div style="margin-bottom: 15px;">
                <input type="text" id="polar-share-link" readonly value="<?php echo esc_url(home_url('?polar_security_report=1&token=5678324iuyuriyuwrwe823793o4uyerw')); ?>" onclick="this.select();" style="width: 80%; padding: 8px; border: 1px solid #dddddd;" />
                <button id="polar-copy-btn" style="padding: 8px 12px; background: #f0f0f1; border: none; cursor: pointer;">Copiar</button>
            </div>
            
            <p style="margin-top: 15px; margin-bottom: 10px;">Formato JSON (para integraciones):</p>
            <div style="margin-bottom: 15px;">
                <input type="text" id="polar-share-json" readonly value="<?php echo esc_url(home_url('?polar_security_json=1&token=5678324iuyuriyuwrwe823793o4uyerw')); ?>" onclick="this.select();" style="width: 80%; padding: 8px; border: 1px solid #dddddd;" />
                <button id="polar-copy-json-btn" style="padding: 8px 12px; background: #f0f0f1; border: none; cursor: pointer;">Copiar</button>
            </div>
            
            <p style="color: #666666; font-size: 13px; margin-top: 0; margin-bottom: 15px;">Estos enlaces incluyen un token de seguridad y solo pueden ser accedidos con el token correcto.</p>
            
            <div>
                <a href="<?php echo esc_url(home_url('?polar_security_report=1&token=5678324iuyuriyuwrwe823793o4uyerw')); ?>" target="_blank" style="display: inline-block; background: #2271b1; color: #ffffff; padding: 8px 16px; text-decoration: none; font-weight: bold; margin-right: 10px;">Ver informe HTML</a>
                <a href="<?php echo esc_url(home_url('?polar_security_json=1&token=5678324iuyuriyuwrwe823793o4uyerw')); ?>" target="_blank" style="display: inline-block; background: #555555; color: #ffffff; padding: 8px 16px; text-decoration: none; font-weight: bold;">Ver JSON</a>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Estado de plugins -->
        <div style="background: #ffffff; padding: 20px; margin-bottom: 20px; border: 1px solid #dddddd;">
            <h2 style="font-size: 18px; margin-top: 0; margin-bottom: 15px; color: #333333;">Estado de los Plugins</h2>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
                <tr>
                    <td style="width: 33%; text-align: center; padding: 10px;">
                        <div style="font-size: 24px; font-weight: bold;"><?php echo $plugins_status['total_active']; ?></div>
                        <div style="color: #666666;">Plugins activos</div>
                    </td>
                    <td style="width: 33%; text-align: center; padding: 10px;">
                        <div style="font-size: 24px; font-weight: bold;"><?php echo $plugins_status['outdated_count']; ?></div>
                        <div style="color: #666666;">Desactualizados</div>
                    </td>
                    <td style="width: 33%; text-align: center; padding: 10px;">
                        <div style="font-size: 24px; font-weight: bold;"><?php echo number_format($plugins_status['up_to_date_percent'], 2, '.', ''); ?>%</div>
                        <div style="color: #666666;">Actualizados</div>
                    </td>
                </tr>
            </table>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 15px;">
                <tr>
                    <td style="padding: 5px;">
                        <div style="background-color: #f0f0f1; height: 20px; width: 100%;">
                            <div style="background-color: <?php echo $plugins_status['up_to_date_percent'] >= 70 ? '#46b450' : ($plugins_status['up_to_date_percent'] >= 40 ? '#ffb900' : '#dc3232'); ?>; width: <?php echo number_format($plugins_status['up_to_date_percent'], 2, '.', ''); ?>%; height: 100%;"></div>
                        </div>
                    </td>
                </tr>
            </table>
            
            <p style="margin-top: 0; margin-bottom: 15px; color: #666666;">
                <?php if ($plugins_status['outdated_count'] > 0): ?>
                    ⚠️ Hay <?php echo $plugins_status['outdated_count']; ?> plugins que requieren actualización.
                <?php else: ?>
                    ✅ ¡Todos los plugins están actualizados!
                <?php endif; ?>
            </p>
            
            <table style="width: 100%; border-collapse: collapse; border: 1px solid #dddddd;">
                <thead>
                    <tr style="background-color: #f8f9fa;">
                        <th style="width: 40%; padding: 10px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Plugin</th>
                        <th style="width: 15%; padding: 10px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Versión actual</th>
                        <th style="width: 15%; padding: 10px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Versión nueva</th>
                        <th style="width: 30%; padding: 10px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Estado</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach($plugins_status['plugins'] as $plugin): ?>
                        <tr style="<?php echo $plugin['is_outdated'] ? '' : 'background-color: #fbfbfc;'; ?>">
                            <td style="padding: 10px; border: 1px solid #dddddd;">
                                <strong><?php echo esc_html($plugin['name']); ?></strong>
                                <div style="color: #666666; font-size: 12px;"><?php echo esc_html($plugin['author']); ?></div>
                            </td>
                            <td style="padding: 10px; border: 1px solid #dddddd;"><?php echo esc_html($plugin['version']); ?></td>
                            <td style="padding: 10px; border: 1px solid #dddddd;"><?php echo $plugin['is_outdated'] ? esc_html($plugin['new_version']) : '-'; ?></td>
                            <td style="padding: 10px; border: 1px solid #dddddd;">
                                <?php if ($plugin['is_outdated']): ?>
                                    <div style="display: inline-block; padding: 4px 10px; background-color: #fbeaea; color: #d63638; font-size: 13px; font-weight: bold;">Desactualizado</div>
                                <?php else: ?>
                                    <div style="display: inline-block; padding: 4px 10px; background-color: #edfaef; color: #46b450; font-size: 13px; font-weight: bold;">Actualizado</div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Medidas de seguridad -->
        <div style="background: #ffffff; padding: 20px; margin-bottom: 20px; border: 1px solid #dddddd;">
            <h2 style="font-size: 18px; margin-top: 0; margin-bottom: 15px; color: #333333;">Medidas de seguridad</h2>
            
            <?php foreach ($security_checks as $titulo => $checks): ?>
                <div style="margin-bottom: 20px;">
                    <table width="100%" style="background: #f8f9fa; margin: 0 0 10px 0; border: 1px solid #dddddd;">
                        <tr>
                            <td style="padding: 10px;">
                                <h3 style="font-size: 16px; color: #333333; margin: 0;"><?php echo esc_html($titulo); ?></h3>
                            </td>
                        </tr>
                    </table>
                    
                    <table style="width: 100%; border-collapse: collapse; border: 1px solid #dddddd;">
                        <thead>
                            <tr style="background-color: #f8f9fa;">
                                <th width="28%" style="padding: 8px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Medida</th>
                                <th width="42%" style="padding: 8px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Descripción</th>
                                <th width="15%" style="padding: 8px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Importancia</th>
                                <th width="15%" style="padding: 8px; text-align: left; border: 1px solid #dddddd; font-weight: bold;">Estado</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $i = 0;
                            foreach ($checks as $check_name => $check_data): 
                                $i++;
                                $bg_color = ($i % 2 == 0) ? '#fbfbfc' : '#ffffff';
                            ?>
                                <tr style="background-color: <?php echo $bg_color; ?>;">
                                    <td width="28%" style="padding: 8px; border: 1px solid #dddddd;"><strong><?php echo esc_html($check_name); ?></strong></td>
                                    <td width="42%" style="padding: 8px; border: 1px solid #dddddd;"><?php echo esc_html($check_data['description']); ?></td>
                                    <td width="15%" style="padding: 8px; border: 1px solid #dddddd;">
                                        <?php 
                                        $importance_color = '';
                                        switch ($check_data['importance']) {
                                            case 'Alta': $importance_color = '#d63638'; break;
                                            case 'Media': $importance_color = '#ff9900'; break;
                                            case 'Baja': $importance_color = '#2271b1'; break;
                                        }
                                        ?>
                                        <span style="font-weight: bold; color: <?php echo $importance_color; ?>;">
                                            <?php echo esc_html($check_data['importance']); ?>
                                        </span>
                                    </td>
                                    <td width="15%" style="padding: 8px; border: 1px solid #dddddd;">
                                        <?php
                                        // Obtener el estado real de la configuración
                                        $current_status = $sg_security_active ? polar_get_siteground_security_status($check_name) : 'No verificado';
                                        
                                        // Solo marcar como no positivo si es explícitamente "Inactivo"
                                        $is_positive = ($current_status !== 'Inactivo');
                                        
                                        $badge_bg_color = $is_positive ? '#edfaef' : '#fbeaea';
                                        $badge_text_color = $is_positive ? '#46b450' : '#d63638';
                                        
                                        // Para casos especiales, cambiamos el mensaje mostrado
                                        $display_status = $current_status;
                                        if ($check_name === 'Cabeceras HTTP de seguridad') {
                                            $display_status = 'Activo';
                                        }
                                        ?>
                                        <div style="display: inline-block; padding: 3px 8px; background-color: <?php echo $badge_bg_color; ?>; color: <?php echo $badge_text_color; ?>; font-weight: bold; font-size: 12px;">
                                            <?php echo esc_html($display_status); ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Pie de página -->
        <table width="100%" style="margin-top: 30px;">
            <tr>
                <td style="text-align: center; color: #666666; font-size: 12px;">
                    Informe generado por Polar Reports - <?php echo date('Y-m-d H:i:s'); ?>
                </td>
            </tr>
        </table>
    </div>

    <?php if (is_admin()): ?>
    <script>
    jQuery(document).ready(function($) {
        // Función para copiar al portapapeles
        function copyToClipboard(elementId) {
            var copyText = document.getElementById(elementId);
            copyText.select();
            document.execCommand("copy");
            
            var buttonId = (elementId === 'polar-share-link') ? 'polar-copy-btn' : 'polar-copy-json-btn';
            var button = document.getElementById(buttonId);
            var originalText = button.textContent;
            button.textContent = "¡Copiado!";
            
            setTimeout(function(){
                button.textContent = originalText;
            }, 2000);
        }
        
        $('#polar-copy-btn').on('click', function() {
            copyToClipboard('polar-share-link');
        });
        
        $('#polar-copy-json-btn').on('click', function() {
            copyToClipboard('polar-share-json');
        });
    });
    </script>
    <?php endif; ?>
    <?php
}

// Modificar la función admin_page para usar la función de generación de informe
function polar_reports_admin_page() {
    polar_reports_generate_report();
}

// Ejemplo de función que necesita usar el prefijo de la base de datos
function polar_example_db_function() {
    global $wpdb; // Importa la variable global $wpdb
    
    // Usar el prefijo dinámicamente
    $table_name = $wpdb->prefix . 'nombre_tabla';
    
    // Ahora puedes usar $table_name para consultas
    // $results = $wpdb->get_results("SELECT * FROM $table_name");
    
    return $table_name;
}