<?php
// Evitar el acceso directo al archivo
if (!defined('ABSPATH')) {
    exit;
}

class Dworkers_AI_Text {
    private static function make_request($api_key, $prompt, $model = 'gpt-4', $max_retries = 3) {
        $provider = get_option('dworkers_content_seo_agent_ai_provider', 'OpenAI');
        $version = get_option('dworkers_content_seo_agent_version', 'free');
        if ($version !== 'free') {
            $token = get_option('dworkers_content_seo_agent_token', '');
            $response = wp_remote_post('https://app.markplanner.co/api/plugin/api_key', array(
                'body' => [
                    'token' => $token
                ]
            ));

            if (!is_wp_error($response)) {
                $body = json_decode(wp_remote_retrieve_body($response), true);
                if (isset($body['api_key'])) {
                    $api_key = $body['api_key'];
                    $provider = 'OpenAI';
                }
            }
        }

        $attempt = 0;
        $timeout = 120;

        while ($attempt < $max_retries) {
            error_log("Intento " . ($attempt + 1) . " de solicitud a OpenAI para prompt: " . substr($prompt, 0, 100) . "...");

            $args = array(
                'timeout' => $timeout,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode(array(
                    'model' => $model,
                    'messages' => array(
                        array('role' => 'user', 'content' => $prompt)
                    ),
                    'temperature' => 0.7
                ))
            );

            error_log("Cuerpo de la solicitud: " . $args['body']);

            if ($provider === 'OpenAI') {
                error_log("Usando OpenAI para la solicitud");
                $response = wp_remote_post('https://api.openai.com/v1/chat/completions', $args);
            } else {
                error_log("Usando Groq para la solicitud");
                $response = wp_remote_post('https://api.groq.com/openai/v1/chat/completions', $args);
            }

            if (is_wp_error($response)) {
                error_log("Error en la solicitud a OpenAI: " . $response->get_error_message());
                $attempt++;
                continue;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);
            $http_code = wp_remote_retrieve_response_code($response);

            error_log("Respuesta de OpenAI. Código HTTP: " . $http_code);
            error_log("Cuerpo de la respuesta: " . print_r($body, true));

            if ($http_code === 200 && isset($body['choices'][0]['message']['content'])) {
                error_log("Contenido generado con éxito");
                return array('success' => true, 'content' => trim($body['choices'][0]['message']['content']));
            } else {
                error_log("Respuesta inesperada de OpenAI. Código HTTP: " . $http_code);
                $attempt++;
            }
        }

        error_log("Se agotaron los intentos de solicitud a OpenAI");
        return array('success' => false, 'message' => 'No se pudo generar el contenido después de ' . $max_retries . ' intentos');
    }

    public static function generate_title($api_key, $model, $tematica) {
        error_log("Iniciando generate_title en " . __FILE__);
        error_log("Generando título para la temática: " . $tematica);
        $prompt = "Genera un título llamativo y creativo, de máximo 64 caracteres incluido espacios, no debe tener más de 64 caracteres, para un post de blog sobre {$tematica} que no tenga mas de 64 caracteres, 64 caracteres máximo, amigable y no muy técnico y que no sea presuntuoso ni con palabras muy rebuscadas y no capitalizes todas las palabras, solo la primera y los nombres propios";
        return self::make_request($api_key, $prompt, $model);
    }

    public static function generate_keyword($api_key, $model, $title) {
        error_log("Iniciando generate_keyword en " . __FILE__);
        $prompt = "Genera una palabra clave para SEO, una palabra o máximo 4 como una sola frase unicamente, a partir del título del blog: {$title}";
        return self::make_request($api_key, $prompt, $model);
    }

    public static function generate_excerpt($api_key, $model, $tematica) {
        error_log("Inicio de generate_excerpt con temática: " . $tematica);
        $prompt = "Genera un extracto breve y atractivo, con alrededor de 130 caractéres, para un artículo sobre: {$tematica}";
        return self::make_request($api_key, $prompt, $model);
    }

    public static function generate_content($api_key, $modelo, $tematica, $title, $keyword) {
        error_log("Inicio de generate_content con temática: " . $tematica . ", modelo: " . $modelo . ", título: " . $title . ", palabra clave: " . $keyword);
        $prompt = "Genera un contenido de blog, Inicia el contenido con un párrafo completo de introducción y sin título, de al menos 1600 palabras sobre el título: \"" . $title . "\", sin incluir este titulo como h1 por favor no lo incluyas, que sea efectivo para SEO incluyendo la palabra clave \"" . $keyword . "\" al menos dos veces, una de ella en el parrafo inicial y otra en un subtítulo h2 o h3 cualquiera. Inicia el contenido con un párrafo. Por favor, incluye títulos y subtítulos en formato HTML (como <h2> o <h3>), no uses finales como: en conclusión, en resumen, no incluyas el titulo principal \"" . $title . "\" al inicio del artículo.

        Para optimización SEO:
        1. Utiliza la palabra clave principal \"" . $keyword . "\" en el primer párrafo y en al menos un subtítulo.
        2. Incluye al menos 2-3 palabras clave secundarias relacionadas con el tema.
        3. Usa la palabra clave principal en el 1-2% del contenido total.
        4. Asegúrate de que los subtítulos (H2, H3) contengan palabras clave relevantes.
        5. Incluye al menos un enlace interno relevante (usa un ejemplo genérico si es necesario).
        6. Escribe párrafos cortos y utiliza listas cuando sea apropiado para mejorar la legibilidad.

        Cuando incluyas código en el contenido, enciérralo en etiquetas de código HTML y especifica el lenguaje. Por ejemplo:

        <pre><code class=\"language-apache\">
        RewriteEngine On
        RewriteCond %{HTTPS} !=on
        RewriteRule ^.*$ https://%{SERVER_NAME}%{REQUEST_URI} [R=301,L]
        </code></pre>

        Adicional, al final del artículo, agrega un enlace externo relevante utilizando el siguiente formato HTML exacto: <a href=\"URL_DEL_SITIO\" target=\"_blank\" rel=\"noopener noreferrer\">TEXTO_DEL_ENLACE</a>. Asegúrate de que este enlace sea a una web real y relevante, no uses enlaces de ejemplo o inexistentes.";

        $content = self::make_request($api_key, $prompt, $modelo);
        $category = self::generate_category($api_key, $modelo, $title, $content['content']);

        $random_post = get_posts(array(
            'post_type' => 'post',
            'posts_per_page' => 1,
            'orderby' => 'rand'
        ));

        if (!empty($random_post)) {
            $random_post_link = get_permalink($random_post[0]->ID);
            error_log("Enlace del post aleatorio: " . $random_post_link);
        } else {
            $random_post_link = '';
            error_log("No se encontró ningún post aleatorio.");
        }

        $content = $content['content'] . "<p>¿Ya leiste nuestro artículo: <a href='$random_post_link'>$random_post_link</a>?</p>";

        $token = get_option('dworkers_content_seo_agent_token', '');
        wp_remote_post('https://app.markplanner.co/api/plugin/token/usage', array(
            'body' => [
                'token' => $token
            ]
        ));

        return [
            'success' => true,
            'title' => $title,
            'content' => $content,
            'keyword' => $keyword,
            'category' => $category
        ];
    }

    public static function generate_category($api_key, $model, $title, $content) {
        $prompt = "Basándote en el siguiente título y extracto de contenido, sugiere una categoría apropiada de una o dos palabras para este artículo:\n\nTítulo: $title\n\nContenido: " . substr($content, 0, 500) . "...";

        $response = self::make_request($api_key, $prompt, $model);

        if ($response['success']) {
            $generated_category = trim($response['content']);
            error_log("Categoría completa generada por GPT: " . $generated_category);

            // Limitar la categoría a dos palabras
            $words = explode(' ', $generated_category);
            $short_category = implode(' ', array_slice($words, 0, 2));
            error_log("Categoría acortada: " . $short_category);

            // Obtener categorías existentes en WordPress
            $existing_categories = get_categories(array('hide_empty' => false));
            $category_names = wp_list_pluck($existing_categories, 'name');
            $category_id = 0;

            // Crear prompt para comparar con GPT las categorías existentes
            $comparison_prompt = "Compara la categoría generada: \"$short_category\", con las siguientes categorías existentes: " . implode(', ', $category_names) . ". No hagas una comparación textual exacta, sino semántica. Si la categoría generada es similar en significado a cualquiera de las existentes, devuelve un JSON con dos campos: 'new'=false y 'name'=El nombre de la categoría existente a la cual se relacionó. De lo contrario, retorno un JSON con un solo campo: 'new'=true. No generes más texto adicional, ni uses un formato especial como MARKDOWN, solo escribe la estructura JSON especificada.";

            // Solicitud a GPT para comparar
            $comparison_response = self::make_request($api_key, $comparison_prompt, $model);

            if ($comparison_response['success']) {
                $comparison_response['content'] = str_replace('```', '', $comparison_response['content']);
                $comparison_response['content'] = str_replace('json', '', $comparison_response['content']);
                $comparison_data = json_decode($comparison_response['content'], true);

                if (json_last_error() === JSON_ERROR_NONE && isset($comparison_data['new']) && isset($comparison_data['name'])) {
                    // Si existe la categoría o si es nueva, manejar la respuesta
                    if (!$comparison_data['new']) {
                        $category_id = get_cat_ID($comparison_data['name']);
                        $short_category = $comparison_data['name'];
                    } else {
                        // Category does not exist, create it
                        $result = wp_insert_term(
                            $short_category,   // the term
                            'category',       // the taxonomy
                            array(
                                'description' => '',
                                'slug'        => ''
                            )
                        );

                        if (is_wp_error($result)) {
                            error_log('Error creating category: ' . $result->get_error_message());
                        } else {
                            $category_id = $result['term_id'];
                            error_log('Category created successfully: ' . $short_category);
                        }
                    }
                    return array(
                        'name' => $short_category,
                        'full_name' => $generated_category,
                        'new' => $comparison_data['new'],
                        'id' => $category_id,
                    );
                } else {
                    error_log("Error en la estructura del JSON devuelto por GPT o campos faltantes.");
                }
            } else {
                error_log("Error en la comparación de categorías con GPT.");
            }

            error_log("Nueva categoría sugerida manualmente: " . $short_category);

            $category = get_term_by('name', $short_category, 'category');
            if (!$category) {
                // Category does not exist, create it
                $result = wp_insert_term(
                    $short_category,   // the term
                    'category',       // the taxonomy
                    array(
                        'description' => '',
                        'slug'        => ''
                    )
                );

                if (is_wp_error($result)) {
                    error_log('Error creating category: ' . $result->get_error_message());
                } else {
                    $category_id = $result['term_id'];
                    error_log('Category created successfully: ' . $short_category);
                }
            } else {
                $category_id = $category->term_id;
                error_log('Category already exists: ' . $short_category);
            }

            return array(
                'name' => $short_category,
                'full_name' => $generated_category,
                'new' => true,
                'id' => $category_id,
            );
        } else {
            error_log("Error al generar categoría: " . print_r($response, true));
            return array(
                'name' => 'Sin categoría',
                'full_name' => 'Sin categoría',
                'new' => true,
                'id' => 0
            );
        }
    }
}
