<?php
/*
Plugin Name: Dworkers Content and SEO Agent
Description: Este plugin crea contenido en WordPress y es 100% SEO friendly para YOAST, incluyendo palabras clave, categorías, y más.
Author: Dworkers
Author URI: https://workers.zeroazul.com
Version: 1.0
*/

// Evitar el acceso directo
if (!defined('ABSPATH')) {
    exit;
}

// Incluir los archivos necesarios
require_once plugin_dir_path(__FILE__) . 'includes/write-content.php'; // Mueve esto fuera de la función
require_once plugin_dir_path(__FILE__) . 'includes/ai-comment.php';
require_once plugin_dir_path(__FILE__) . 'includes/ai-text.php';
require_once plugin_dir_path(__FILE__) . 'includes/ai-image.php';
require_once plugin_dir_path(__FILE__) . 'includes/ai-seo.php';

// Inicializar el plugin
add_action( 'plugins_loaded', 'dworkers_init' );

// Registrar página de administración
add_action('admin_menu', 'dworkers_content_seo_agent_add_admin_page');

function dworkers_content_seo_agent_add_admin_page() {
    add_menu_page(
        'MARK Content Generator', // Título de la página
        'MARK Content Generator', // Título del menú
        'manage_options', // Capacidad requerida
        'dworkers-content-seo-agent', // Slug del menú
        'dworkers_content_seo_agent_display_admin_page', // Función que renderiza la página
        'dashicons-admin-generic', // Icono
        110 // Posición en el menú
    );
}

function dworkers_content_seo_agent_display_admin_page() {
    $tab = isset($_GET['tab']) ? $_GET['tab'] : 'post';
    if ($tab === 'post') {
        echo '<div id="dworkers-root-post"></div>';
        return;
    }
    if ($tab === 'comment') {
        echo '<div id="dworkers-root-comment"></div>';
        return;
    }
    if ($tab === 'settings') {
        echo '<div id="dworkers-root-settings"></div>';
        return;
    }
}

function dworkers_content_seo_agent_generate_post_content(WP_REST_Request $request) {
    try {
        $api_key = get_option('dworkers_content_seo_agent_api_key', '');
        $tematica = $request->get_param('tematica');
        $modelo = get_option('dworkers_content_seo_agent_ai_model');

        $result = array();

        // Generar título
        $title_result = Dworkers_AI_Text::generate_title($api_key, $modelo, $tematica);
        if (!$title_result['success']) {
            throw new Exception('Error al generar título: ' . $title_result['message']);
        }
        $result['title'] = str_replace('"', '', $title_result['content']);

        // Generar palabra clave
        $keyword_result = Dworkers_AI_Text::generate_keyword($api_key, $modelo, $result['title']);
        if (!$keyword_result['success']) {
            throw new Exception('Error al generar palabra clave: ' . $keyword_result['message']);
        }
        $result['keyword'] = str_replace('"', '', $keyword_result['content']);

        // Generar extracto
        $excerpt_result = Dworkers_AI_Text::generate_excerpt($api_key, $modelo, $tematica);
        if (!$excerpt_result['success']) {
            throw new Exception('Error al generar extracto: ' . $excerpt_result['message']);
        }
        $result['excerpt'] = str_replace('"', '', $excerpt_result['content']);

        // Generar contenido
        $content_result = Dworkers_AI_Text::generate_content($api_key, $modelo, $tematica, $result['title'], $result['keyword']);
        if (!$content_result['success']) {
            throw new Exception('Error al generar contenido: ' . $content_result['message']);
        }
        $result['content'] = $content_result['content'];

        // Incluir la categoría generada
        if (isset($content_result['category'])) {
            $category_name = $content_result['category']['name']; // Obtener el nombre de la categoría
            $is_new = $content_result['category']['new']; // Verificar si es nueva o existente
            $category_id = $content_result['category']['id'];

            // Depurar si el ID de la categoría es válido
            error_log("Categoría generada: $category_name, ID de la categoría: $category_id, Nueva: " . ($is_new ? 'Sí' : 'No'));

            $result['category'] = $category_name . ($is_new ? ' (Nueva categoría generada)' : ' (Categoría existente)');
            $result['category_id'] = $category_id;
        }

        wp_send_json_success($result);

    } catch (Exception $e) {
        error_log("Error en la generación de contenido: " . $e->getMessage());
        wp_send_json_error(array('message' => 'Error: ' . $e->getMessage()));
    }
}


function dworkers_content_seo_agent_generate_image(WP_REST_Request $request) {
    $api_key = get_option('dworkers_content_seo_agent_api_key', '');
    $title = $request->get_param('title');

    $result = Dworkers_AI_Image::generate_image($api_key, $title);

    if ($result['success']) {
        wp_send_json_success($result);
    } else {
        wp_send_json_error($result);
    }
}

function dworkers_content_seo_agent_generate_image_alt(WP_REST_Request $request) {
    $api_key = get_option('dworkers_content_seo_agent_api_key', '');
    $title = $request->get_param('title');
    $model = get_option('dworkers_content_seo_agent_ai_model');

    $result = Dworkers_AI_Image::generate_image_alt($api_key, $model, $title);

    if ($result['success']) {
        wp_send_json_success($result);
    } else {
        wp_send_json_error($result);
    }
}

function dworkers_content_seo_agent_generate_seo_info(WP_REST_Request $request) {
    $api_key = get_option('dworkers_content_seo_agent_api_key', '');
    $title = $request->get_param('title');
    $model = get_option('dworkers_content_seo_agent_ai_model');

    $result = Dworkers_AI_SEO::generate_seo_image_name($api_key, $model, $title);

    if ($result['success']) {
        wp_send_json_success($result);
    } else {
        wp_send_json_error($result);
    }
}

function dworkers_content_seo_agent_generate_comment(WP_REST_Request $request) {
    $api_key = get_option('dworkers_content_seo_agent_api_key', '');
    $model = get_option('dworkers_content_seo_agent_ai_model');
    $style = $request->get_param('style');
    $post_id = $request->get_param('post_id');

    try {
        if ($post_id === "") {
            $post_id = Dworkers_AI_Comment::get_random_post();
        }

        // Get the post object
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception('Post no encontrado');
        }

        // Get the post title and excerpt
        $title = get_the_title($post);
        $excerpt = get_the_excerpt($post);

        $comment = Dworkers_AI_Comment::generate_comment($api_key, $model, $style, $title, $excerpt);
        if (!$comment['success']) {
            throw new Exception('Error al generar comentario: ' . $comment['message']);
        }

        $name = Dworkers_AI_Comment::generate_name($api_key, $model);
        if (!$name['success']) {
            throw new Exception('Error al generar nombre: ' . $name['message']);
        }

        $email = Dworkers_AI_Comment::generate_email($api_key, $model, $name['content']);
        if (!$email['success']) {
            throw new Exception('Error al generar email: ' . $email['message']);
        }

        $response = dworkers_save_comment($post_id, $comment['content'], $name['content'], $email['content']);

        wp_send_json_success([
            'post_id' => $post_id,
            'comment' => $comment['content'],
            'name' => $name['content'],
            'email' => $email['content'],
            'message' => $response['message'],
        ]);

    } catch (Exception $e) {
        error_log("Error en la generación de contenido: " . $e->getMessage());
        wp_send_json_error(array('message' => 'Error: ' . $e->getMessage()));
    }
}

function dworkers_get_posts() {
    $args = array(
        'post_type' => 'post',
        'post_status' => 'publish',
        'posts_per_page' => -1,
    );

    $query = new WP_Query($args);
    $posts = array();

    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();
            $title = str_replace('&#8221;', '', get_the_title());
            $title = str_replace('&#8220;', '', $title);
            $posts[] = array(
                'id' => get_the_ID(),
                'title' => $title,
            );
        }
        wp_reset_postdata();
    }

    wp_send_json_success($posts);
}

function dworkers_init() {
	if ( !class_exists( 'WPSEO_Frontend' ) ) {
		add_action( 'admin_notices', 'wpseo_not_loaded' );
	} else {
		include __DIR__ . '/classes/class-wpseo-frontend-to-rest-api.php';
		include __DIR__ . '/classes/class-wpseo-to-rest-api.php';
        $yoast_To_REST_API = new Yoast_To_REST_API();

        // Registrar CORS
        add_action('init','add_cors_http_header');

        add_action('admin_enqueue_scripts', function () {
            // Enqueue the CSS file
            wp_enqueue_style(
                'dworkers-react-app-css',
                plugins_url('/admin/assets/app.css', __FILE__),
                array(),
                '1.0.0'
            );

            // Enqueue the JavaScript file
            wp_enqueue_script(
                'dworkers-react-app-js',
                plugins_url('/admin/assets/app.js', __FILE__),
                [],
                '1.0.0',
                true
            );

            // Ensure the script is treated as a module
            wp_script_add_data('dworkers-react-app-js', 'type', 'module');
        });

        // Registrar las rutas de API
        add_action('rest_api_init', function () {
            register_rest_route('dworkers/v1', '/generate/post', array(
                'methods' => 'POST',
                'callback' => 'dworkers_content_seo_agent_generate_post_content',
                'permission_callback' => '__return_true',
            ));
            register_rest_route('dworkers/v1', '/generate/image', array(
                'methods' => 'POST',
                'callback' => 'dworkers_content_seo_agent_generate_image',
                'permission_callback' => '__return_true',
            ));
            register_rest_route('dworkers/v1', '/generate/alt', array(
                'methods' => 'POST',
                'callback' => 'dworkers_content_seo_agent_generate_image_alt',
                'permission_callback' => '__return_true',
            ));
            register_rest_route('dworkers/v1', '/generate/seo', array(
                'methods' => 'POST',
                'callback' => 'dworkers_content_seo_agent_generate_seo_info',
                'permission_callback' => '__return_true',
            ));
            register_rest_route('dworkers/v1', '/generate/comment', array(
                'methods' => 'POST',
                'callback' => 'dworkers_content_seo_agent_generate_comment',
                'permission_callback' => '__return_true',
            ));
            register_rest_route('dworkers/v1', '/save/post', array(
                'methods' => 'POST',
                'callback' => 'dworkers_save_post',
                'permission_callback' => '__return_true',
            ));
            register_rest_route('dworkers/v1', '/posts', array(
                'methods' => 'GET',
                'callback' => 'dworkers_get_posts',
                'permission_callback' => '__return_true',
            ));
        });

        // Registrar las rutas de la API admin
        include __DIR__ . '/admin/api/settings.php';
    }
}

function add_cors_http_header(){
    header("Access-Control-Allow-Origin: *");
}

function wpseo_not_loaded() {
	printf(
		'<div class="error"><p>%s</p></div>',
		__( '<b>Dworkers Content and SEO Agent</b> plugin not working because <b>Yoast SEO</b> plugin is not active.' )
	);
}

